<?php
/**
 * Add form MailChimp action.
 *
 * @package Raven
 * @since 1.0.0
 */

namespace Raven\Modules\Forms\Actions;

defined( 'ABSPATH' ) || die();

use Raven\Modules\Forms\Classes\MailChimp_Handler;
use Elementor\Settings;

/**
 * MailChimp Action.
 *
 * Initializing the MailChimp action by extending action base.
 *
 * @since 1.0.0
 */
class Mailchimp extends Action_Base {

	/**
	 * Update controls.
	 *
	 * Add MailChimp section.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param object $widget Widget instance.
	 */
	public function update_controls( $widget ) {

		$widget->start_controls_section(
			'section_mailchimp',
			[
				'label' => __( 'MailChimp', 'raven' ),
				'condition' => [
					'actions' => 'mailchimp',
				],
			]
		);

		if ( empty( get_option( 'elementor_raven_mailchimp_api_key' ) ) ) {
			$widget->add_control(
				'_api_key_msg',
				[
					'type' => 'raw_html',
					'raw' => sprintf(
						/* translators: %s: Settings page URL */
						__( 'Set your MailChimp API in <a target="_blank" href="%s">Raven Settings <i class="fa fa-external-link-square"></i></a>.', 'raven' ),
						Settings::get_url() . '#tab-raven'
					),
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-danger',
				]
			);

			return $widget->end_controls_section();
		}

		$widget->add_control(
			'mailchimp_list',
			[
				'label' => __( 'List', 'raven' ),
				'type' => 'select',
				'frontend_available' => true,
				'render_type' => 'ui',
			]
		);

		$widget->add_control(
			'mailchimp_field_mapping',
			[
				'label' => __( 'Field Mapping', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$widget->add_control(
			'mailchimp_field_mapping_email',
			[
				'label' => __( 'Email', 'raven' ),
				'type' => 'text',
				'type' => 'select',
				'render_type' => 'ui',
			]
		);

		$widget->add_control(
			'mailchimp_field_mapping_first_name',
			[
				'label' => __( 'First Name', 'raven' ),
				'type' => 'select',
				'render_type' => 'ui',
			]
		);

		$widget->add_control(
			'mailchimp_field_mapping_last_name',
			[
				'label' => __( 'Last Name', 'raven' ),
				'type' => 'select',
				'render_type' => 'ui',
			]
		);

		$widget->end_controls_section();
	}

	/**
	 * Run action.
	 *
	 * Subscribe email to MailChimp.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 *
	 * @param object $ajax_handler Ajax handler instance.
	 */
	public static function run( $ajax_handler ) {
		$api_key = get_option( 'elementor_raven_mailchimp_api_key' );
		$list_id = $ajax_handler->form['settings']['mailchimp_list'];

		$email_field_id = $ajax_handler->form['settings']['mailchimp_field_mapping_email'];
		$email          = $ajax_handler->record['fields'][ $email_field_id ];

		$first_name_field_id = $ajax_handler->form['settings']['mailchimp_field_mapping_first_name'];
		$first_name          = $ajax_handler->record['fields'][ $first_name_field_id ];

		$last_name_field_id = $ajax_handler->form['settings']['mailchimp_field_mapping_last_name'];
		$last_name          = $ajax_handler->record['fields'][ $last_name_field_id ];

		if ( empty( $list_id ) ) {
			return $ajax_handler->add_response(
				'admin_errors',
				__( 'MailChimp list ID is missing.', 'raven' )
			);
		}

		if ( empty( $email_field_id ) ) {
			return $ajax_handler->add_response(
				'admin_errors',
				__( 'MailChimp Email Field ID is missing.', 'raven' )
			);
		}

		if ( empty( $first_name ) ) {
			$first_name = '';
		}

		if ( empty( $last_name ) ) {
			$last_name = '';
		}

		$handler = new MailChimp_Handler( $api_key );

		$result = $handler->get( "lists/$list_id" );

		if ( ! $handler->success() ) {
			return $ajax_handler->add_response(
				'admin_errors',
				__( 'MailChimp list ID is not valid.', 'raven' )
			);
		}

		$result = $handler->post( "lists/$list_id/members", [
			'email_address' => $email,
			'merge_fields' => [
				'FNAME' => $first_name,
				'LNAME' => $last_name,
			],
			'status' => 'subscribed',
		]);

		if ( $handler->success() ) {
			return;
		}

		return $ajax_handler->add_response(
			'admin_errors',
			$handler->getLastError()
		);
	}

	/**
	 * Get lists.
	 *
	 * Get lists from MailChimp.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 *
	 * @param object $ajax_handler Ajax handler instance.
	 */
	public static function get_list( $ajax_handler ) {
		$api_key = get_option( 'elementor_raven_mailchimp_api_key' );

		if ( empty( $api_key ) ) {
			return $ajax_handler->add_response(
				'error',
				__( 'Global MailChimp API is missing.', 'Raven' )
			);
		}

		$handler = new MailChimp_Handler( $api_key );

		$lists = $handler->get( 'lists' );

		if ( $handler->success() ) {
			return $ajax_handler->add_response( 'success', $lists );
		}

		return $ajax_handler->add_response( 'error', $handler->getLastError() );
	}

	/**
	 * Register admin fields.
	 *
	 * Register required admin settings for the field.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param object $settings Settings.
	 */
	public function register_admin_fields( $settings ) {
		$settings->add_section( 'raven', 'raven_mailchimp', [
			'callback' => function() {
				echo '<hr><h2>' . esc_html__( 'MailChimp', 'raven' ) . '</h2>';
			},
			'fields' => [
				'raven_mailchimp_api_key' => [
					'label' => __( 'API Key', 'raven' ),
					'field_args' => [
						'type' => 'text',
						/* translators: %s: MailChimp knowledge base URL  */
						'desc' => sprintf( __( 'To integrate with our forms you need an <a href="%s" target="_blank">API key</a>.', 'raven' ), 'https://kb.mailchimp.com/integrations/api-integrations/about-api-keys' ),
					],
				],
			],
		] );
	}
}
