<?php
namespace Raven\Modules\Forms\Widgets;

use Raven\Base\Base_Widget;
use Raven\Modules\Forms\Module;

defined( 'ABSPATH' ) || die();

class Form extends Base_Widget {

	public function get_name() {
		return 'raven-form';
	}

	public function get_title() {
		return __( 'Form', 'raven' );
	}

	public function get_icon() {
		return 'raven-element-icon raven-element-icon-form';
	}

	protected function _register_controls() {
		$this->register_section_form_fields();
		$this->register_section_submit_button();
		$this->register_section_settings();
		$this->register_section_messages();
		$this->register_section_general();
		$this->register_section_label();
		$this->register_section_field();
		$this->register_section_button();
	}

	private function register_section_form_fields() {

		$field_types = [
			'text' => __( 'Text', 'raven' ),
			'email' => __( 'Email', 'raven' ),
			'textarea' => __( 'Textarea', 'raven' ),
			'tel' => __( 'Tel', 'raven' ),
			'number' => __( 'Number', 'raven' ),
			'acceptance' => __( 'Acceptance', 'raven' ),
			'recaptcha' => __( 'reCAPTCHA', 'raven' ),
		];

		$this->start_controls_section(
			'section_form_fields',
			[
				'label' => __( 'Form Fields', 'raven' ),
			]
		);

		$this->add_control(
			'form_name',
			[
				'label' => __( 'Form', 'raven' ),
				'type' => 'text',
				'default' => 'New form',
				'placeholder' => __( 'Enter your form name', 'raven' ),
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'_id',
			[
				'label' => __( 'ID', 'raven' ),
				'type' => 'hidden',
			]
		);

		$repeater->add_control(
			'type',
			[
				'label' => __( 'Type', 'raven' ),
				'type' => 'select',
				'options' => $field_types,
				'default' => 'text',
			]
		);

		$repeater->add_control(
			'label',
			[
				'label' => __( 'Label', 'raven' ),
				'type' => 'text',
			]
		);

		$repeater->add_control(
			'placeholder',
			[
				'label' => __( 'Placeholder', 'raven' ),
				'type' => 'text',
				'condition' => [
					'type!' => 'acceptance',
					'type!' => 'recaptcha',
				],
			]
		);

		$repeater->add_control(
			'required',
			[
				'label' => __( 'Required', 'raven' ),
				'type' => 'switcher',
				'return_value' => 'true',
				'condition' => [
					'type!' => 'recaptcha',
				],
			]
		);

		$repeater->add_responsive_control(
			'width',
			[
				'label' => __( 'Column Width', 'raven' ),
				'type' => 'select',
				'options' => [
					'' => __( 'Default', 'raven' ),
					'100' => '100%',
					'80' => '80%',
					'75' => '75%',
					'66' => '66%',
					'60' => '60%',
					'50' => '50%',
					'40' => '40%',
					'33' => '33%',
					'25' => '25%',
					'20' => '20%',
				],
				'default' => '100',
				'condition' => [
					'type!' => 'recaptcha',
				],
			]
		);

		$this->add_control(
			'fields',
			[
				'type' => 'repeater',
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'label' => 'Name',
						'type' => 'text',
						'placeholder' => 'Name',
					],
					[
						'label' => 'Email',
						'type' => 'email',
						'placeholder' => 'Email',
						'required' => 'true',
					],
					[
						'label' => 'Message',
						'type' => 'textarea',
						'placeholder' => 'Message',
					],
				],
				'frontend_available' => true,
				'title_field' => '{{{ label }}}',
			]
		);

		$this->end_controls_section();
	}

	private function register_section_submit_button() {
		$this->start_controls_section(
			'section_submit_button',
			[
				'label' => __( 'Submit Button', 'raven' ),
			]
		);

		$this->add_control(
			'submit_button_text',
			[
				'label' => __( 'Text', 'raven' ),
				'type' => 'text',
				'default' => __( 'Send', 'raven' ),
			]
		);

		$this->add_control(
			'submit_button_icon',
			[
				'label' => __( 'Icon', 'raven' ),
				'type' => 'icon',
			]
		);

		$this->add_responsive_control(
			'submit_button_width',
			[
				'label' => __( 'Column Width', 'raven' ),
				'type' => 'select',
				'options' => [
					'' => __( 'Default', 'raven' ),
					'100' => '100%',
					'80' => '80%',
					'75' => '75%',
					'66' => '66%',
					'60' => '60%',
					'50' => '50%',
					'40' => '40%',
					'33' => '33%',
					'25' => '25%',
					'20' => '20%',
				],
				'default' => '100',
			]
		);

		$this->end_controls_section();
	}

	private function register_section_settings() {

		$action_types = [
			'email' => __( 'Email', 'raven' ),
			'mailchimp' => __( 'MailChimp', 'raven' ),
			'redirect' => __( 'Redirect', 'raven' ),
		];

		$this->start_controls_section(
			'section_settings',
			[
				'label' => __( 'Settings', 'raven' ),
			]
		);

		$this->add_control(
			'label',
			[
				'label' => __( 'Label', 'raven' ),
				'type' => 'switcher',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
				'default' => 'yes',
			]
		);

		$this->add_control(
			'required_mark',
			[
				'label' => __( 'Required Mark', 'raven' ),
				'type' => 'switcher',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->add_control(
			'actions',
			[
				'label' => __( 'Add Action', 'raven' ),
				'type' => 'select2',
				'multiple' => true,
				'options' => $action_types,
				'label_block' => true,
				'render_type' => 'ui',
			]
		);

		$this->end_controls_section();
	}

	private function register_section_messages() {
		$this->start_controls_section(
			'section_messages',
			[
				'label' => __( 'Feedback Messages', 'raven' ),
			]
		);

		$this->add_control(
			'messages_custom',
			[
				'label' => __( 'Custom Messages', 'raven' ),
				'type' => 'switcher',
				'render_type' => 'ui',
			]
		);

		$this->add_control(
			'messages_success',
			[
				'label' => __( 'Success Message', 'raven' ),
				'type' => 'text',
				'default' => Module::$messages['success'],
				'label_block' => true,
				'render_type' => 'ui',
				'condition' => [
					'messages_custom' => 'yes',
				],
			]
		);

		$this->add_control(
			'messages_error',
			[
				'label' => __( 'Error Message', 'raven' ),
				'type' => 'text',
				'default' => Module::$messages['error'],
				'label_block' => true,
				'render_type' => 'ui',
				'condition' => [
					'messages_custom' => 'yes',
				],
			]
		);

		$this->add_control(
			'messages_required',
			[
				'label' => __( 'Required Message', 'raven' ),
				'type' => 'text',
				'default' => Module::$messages['required'],
				'label_block' => true,
				'render_type' => 'ui',
				'condition' => [
					'messages_custom' => 'yes',
				],
			]
		);

		$this->add_control(
			'messages_subscriber',
			[
				'label' => __( 'Subscriber Already Exists Message', 'raven' ),
				'type' => 'text',
				'default' => Module::$messages['subscriber'],
				'label_block' => true,
				'render_type' => 'ui',
				'condition' => [
					'messages_custom' => 'yes',
				],
			]
		);

		$this->end_controls_section();
	}


	private function register_section_general() {
		$this->start_controls_section(
			'section_style_general',
			[
				'label' => __( 'General', 'raven' ),
				'tab' => 'style',
			]
		);

		$this->add_responsive_control(
			'general_column_spacing',
			[
				'label' => __( 'Column Spacing', 'raven' ),
				'type' => 'slider',
				'default' => [
					'size' => 7,
				],
				'selectors' => [
					'{{WRAPPER}} .raven-field-group' => 'padding-left: calc( {{SIZE}}{{UNIT}} / 2 );padding-right: calc( {{SIZE}}{{UNIT}} / 2 );',
					'{{WRAPPER}} .raven-form' => 'margin-left: calc( -{{SIZE}}{{UNIT}} / 2 );margin-right: calc( -{{SIZE}}{{UNIT}} / 2 );',
				],
			]
		);

		$this->add_responsive_control(
			'general_row_spacing',
			[
				'label' => __( 'Row Spacing', 'raven' ),
				'type' => 'slider',
				'default' => [
					'size' => 7,
				],
				'selectors' => [
					'{{WRAPPER}} .raven-field-group' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	private function register_section_label() {
		$this->start_controls_section(
			'section_style_label',
			[
				'label' => __( 'Label', 'raven' ),
				'tab' => 'style',
			]
		);

		$this->add_control(
			'label_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-field-label' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			'typography',
			[
				'name' => 'label_typography',
				'selector' => '{{WRAPPER}} .raven-field-label',
				'scheme' => '3',
			]
		);

		$this->add_responsive_control(
			'label_spacing',
			[
				'label' => __( 'Spacing', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-field-group > .raven-field-label' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	private function register_section_field() {
		$this->start_controls_section(
			'section_style_field',
			[
				'label' => __( 'Field', 'raven' ),
				'tab' => 'style',
			]
		);
		$this->start_controls_tabs( 'field_tabs_state' );

		$this->start_controls_tab(
			'field_tab_state_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->add_control(
			'field_tab_background_color_normal',
			[
				'label' => __( 'Background Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-field' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			'border',
			[
				'name' => 'field_tab_border_normal',
				'selector' => '{{WRAPPER}} .raven-field',
			]
		);

		$this->add_responsive_control(
			'field_tab_border_radius_normal',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-field' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			'box-shadow',
			[
				'name' => 'field_tab_box_shadow_normal',
				'exclude' => [
					'box_shadow_position',
				],
				'separator' => 'before',
				'selector' => '{{WRAPPER}} .raven-field',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'field_tab_state_focus',
			[
				'label' => __( 'Focus', 'raven' ),
			]
		);

		$this->add_control(
			'field_tab_background_color_focus',
			[
				'label' => __( 'Background Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-field:focus' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			'border',
			[
				'name' => 'field_tab_border_focus',
				'selector' => '{{WRAPPER}} .raven-field:focus',
			]
		);

		$this->add_responsive_control(
			'field_tab_border_radius_focus',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-field:focus' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			'box-shadow',
			[
				'name' => 'field_tab_box_shadow_focus',
				'exclude' => [
					'box_shadow_position',
				],
				'separator' => 'before',
				'selector' => '{{WRAPPER}} .raven-field:focus',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'field_tabs_separator',
			[
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->start_controls_tabs( 'field_tabs' );

		$this->start_controls_tab(
			'field_tab_placeholder',
			[
				'label' => __( 'Placeholder', 'raven' ),
			]
		);

		$this->add_control(
			'field_tab_color_placeholder',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-field::-webkit-input-placeholder' => 'color: {{VALUE}};',
					'{{WRAPPER}} .raven-field::-ms-input-placeholder' => 'color: {{VALUE}};',
					'{{WRAPPER}} .raven-field::placeholder' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			'typography',
			[
				'name' => 'field_tab_typography_placeholder',
				'selector' => '{{WRAPPER}} .raven-field::placeholder',
				'scheme' => '3',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'field_tab_value',
			[
				'label' => __( 'Value', 'raven' ),
			]
		);

		$this->add_control(
			'field_tab_color_value',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-field' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			'typography',
			[
				'name' => 'field_tab_typography_value',
				'selector' => '{{WRAPPER}} .raven-field',
				'scheme' => '3',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'field_padding',
			[
				'label' => __( 'Padding', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', 'em', '%' ],
				'separator' => 'before',
				'selectors' => [
					'{{WRAPPER}} .raven-field' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	private function register_section_button() {
		$this->start_controls_section(
			'section_style_button',
			[
				'label' => __( 'Button', 'raven' ),
				'tab' => 'style',
			]
		);

		$this->add_responsive_control(
			'button_height',
			[
				'label' => __( 'Height', 'raven' ),
				'type' => 'slider',
				'range' => [
					'px' => [
						'max' => 1000,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-submit-button' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'button_spacing',
			[
				'label' => __( 'Spacing', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-submit-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'button_align',
			[
				'label'  => __( 'Alignment', 'raven' ),
				'type' => 'choose',
				'default' => 'justify',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'raven' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'raven' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'raven' ),
						'icon' => 'fa fa-align-right',
					],
					'justify' => [
						'title' => __( 'Justified', 'raven' ),
						'icon' => 'fa fa-align-justify',
					],
				],
			]
		);

		$this->start_controls_tabs( 'button_tabs' );

		$this->start_controls_tab(
			'button_tab_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->add_control(
			'button_tab_color_normal',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-submit-button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			'typography',
			[
				'name' => 'button_tab_typography_normal',
				'selector' => '{{WRAPPER}} .raven-submit-button',
				'scheme' => '3',
			]
		);

		$this->add_group_control(
			'raven-background',
			[
				'name' => 'button_tab_background_normal',
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .raven-submit-button',
			]
		);

		$this->add_control(
			'button_tab_icon_color_normal',
			[
				'label' => __( 'Icon Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-submit-button i' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'button_tab_hover',
			[
				'label' => __( 'Hover', 'raven' ),
			]
		);

		$this->add_control(
			'button_tab_color_hover',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-submit-button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			'typography',
			[
				'name' => 'button_tab_typography_hover',
				'selector' => '{{WRAPPER}} .raven-submit-button:hover',
				'scheme' => '3',
			]
		);

		$this->add_group_control(
			'raven-background',
			[
				'name' => 'button_tab_background_hover',
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .raven-submit-button:hover',
			]
		);

		$this->add_control(
			'button_tab_icon_color_hover',
			[
				'label' => __( 'Icon Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-submit-button:hover i' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'button_border_heading',
			[
				'label' => __( 'Border', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			'border',
			[
				'name' => 'button_border',
				'selector' => '{{WRAPPER}} .raven-submit-button',
			]
		);

		$this->add_responsive_control(
			'button_radius',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-submit-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			'box-shadow',
			[
				'name' => 'button_box_shadow',
				'exclude' => [
					'box_shadow_position',
				],
				'separator' => 'before',
				'selector' => '{{WRAPPER}} .raven-submit-button',
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_active_settings();
		$fields   = $settings['fields'];

		$this->add_render_attribute( 'form', [
			'class' => 'raven-form raven-flex raven-flex-wrap raven-flex-bottom',
			'method' => 'post',
			'name' => $settings['form_name'],
		] );

		if ( empty( $settings['required_mark'] ) ) {
			$this->add_render_attribute(
				'form',
				'class',
				'raven-hide-required-mark'
			);
		}

		// @todo Move it into a separate method.
		$this->add_render_attribute(
			'submit-button',
			'class',
			'raven-field-group raven-field-type-submit-button elementor-column elementor-col-' . $settings['submit_button_width']
		);

		if ( ! empty( $settings['submit_button_width_tablet'] ) ) {
			$this->add_render_attribute(
				'submit-button',
				'class',
				'elementor-md-' . $settings['submit_button_width_tablet']
			);
		}

		if ( ! empty( $settings['submit_button_width_mobile'] ) ) {
			$this->add_render_attribute(
				'submit-button',
				'class',
				'elementor-sm-' . $settings['submit_button_width_mobile']
			);
		}

		$this->add_render_attribute(
			'submit-button',
			'class',
			'raven-field-align-' . $settings['button_align']
		);

		?>
		<form <?php echo $this->get_render_attribute_string( 'form' ); ?>>
			<input type="hidden" name="post_id" value="<?php echo get_the_ID(); ?>" />
			<input type="hidden" name="form_id" value="<?php echo $this->get_id(); ?>" />
			<?php

			foreach ( $fields as $field ) {
				Module::render_field( $this, $field );
			}

			?>
			<div <?php echo $this->get_render_attribute_string( 'submit-button' ); ?>>
				<button type="submit" class="raven-submit-button">
					<i class="<?php echo $settings['submit_button_icon']; ?>" aria-hidden="true"></i>
					<?php echo $settings['submit_button_text']; ?>
				</button>
			</div>
		</form>
		<?php
	}

}

