<?php
/**
 * Add Template Library Module.
 *
 * @package Raven
 * @since 1.0.0
 */

namespace Raven\Core\Template;

use Elementor\TemplateLibrary;
use Elementor\Api;
use Elementor\Plugin as Elementor;

defined( 'ABSPATH' ) || die();

/**
 * Raven template library module.
 *
 * Raven template library module handler class is responsible for registering and fetching
 * Raven templates.
 *
 * @since 1.0.0
 */
class Module {

	/**
	 * API URL.
	 *
	 * Holds the URL of the API.
	 *
	 * @access private
	 * @static
	 *
	 * @var string API URL.
	 */
	private static $api_url = 'https://demos.artbees.net/jupiterx/library/wp-json/jupiterx/v1/templates/%s';

	/**
	 * Add new categories to list.
	 *
	 * @since 1.0.0
	 *
	 * @param  array $data Template library data including categories and templates.
	 * @return array $data Modified library data.
	 */
	public function add_categories( $data ) {

		$raven_categories   = [ 'maintenance' ];
		$data['categories'] = array_merge( $raven_categories, $data['categories'] );

		sort( $data['categories'] );

		return $data;
	}

	/**
	 * Update template source.
	 *
	 * Elementor has a bug so the `source` should be updated during Ajax call.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function update_source() {
		// phpcs:disable
		if ( ! empty( $_REQUEST['actions'] ) ) {
			if ( false !== strpos( $_REQUEST['actions'], 'get_template_data' ) && false !== strpos( $_REQUEST['actions'], 'raven_' ) ) {
				$_REQUEST['actions'] = str_replace( 'remote', 'raven', $_REQUEST['actions'] );
				$_REQUEST['actions'] = str_replace( 'raven_', '', $_REQUEST['actions'] );
			}
		}
		// phpcs:enable
	}

	/**
	 * Fetch templates from server.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Templates.
	 */
	public static function get_templates() {
		$url = sprintf( self::$api_url, '' );

		$response = wp_remote_get( $url, [
			'timeout' => 40,
		] );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = (int) wp_remote_retrieve_response_code( $response );

		if ( 200 !== $response_code ) {
			return new \WP_Error( 'response_code_error', sprintf( 'The request returned with a status code of %s.', $response_code ) );
		}

		$template_content = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( isset( $template_content['error'] ) ) {
			return new \WP_Error( 'response_error', $template_content['error'] );
		}

		if ( empty( $template_content ) ) {
			return new \WP_Error( 'template_data_error', 'An invalid data was returned.' );
		}

		return $template_content;
	}

	/**
	 * Fetch template content from server.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $template_id Template ID.
	 *
	 * @return array Template content.
	 */
	public static function get_template_content( $template_id ) {
		$url = sprintf( self::$api_url, $template_id );

		$response = wp_remote_get( $url, [
			'timeout' => 60,
		] );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = (int) wp_remote_retrieve_response_code( $response );

		if ( 200 !== $response_code ) {
			return new \WP_Error( 'response_code_error', sprintf( 'The request returned with a status code of %s.', $response_code ) );
		}

		$template_content = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( isset( $template_content['error'] ) ) {
			return new \WP_Error( 'response_error', $template_content['error'] );
		}

		if ( empty( $template_content['content'] ) ) {
			return new \WP_Error( 'template_data_error', 'An invalid data was returned.' );
		}

		$template_content['content'] = json_decode( $template_content['content'], true );

		return $template_content;
	}

	/**
	 * Initialize Raven template library module.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {

		// Add raven categories to the list.
		if ( defined( '\Elementor\Api::LIBRARY_OPTION_KEY' ) ) {
			// Sort before jet elements prepend its categories.
			add_filter( 'option_' . \Elementor\Api::LIBRARY_OPTION_KEY, [ $this, 'add_categories' ], 5 );
		}

		// Register raven source.
		Elementor::$instance->templates_manager->register_source( __NAMESPACE__ . '\Source_Raven' );

		// Update template source.
		add_action( 'wp_ajax_elementor_ajax', [ $this, 'update_source' ], 0 );
	}
}
