<?php
/**
 * Add Post Base.
 *
 * @package Raven
 * @since 1.0.0
 */

namespace Raven\Modules\Posts\Classes;

defined( 'ABSPATH' ) || die();

/**
 * Post Base.
 *
 * An abstract base class to handle controls and render for the post type post.
 *
 * @since 1.0.0
 * @abstract
 *
 * @SuppressWarnings(PHPMD.ExcessiveClassLength)
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
abstract class Post_Base extends Action_Base {

	/**
	 * Register image size control.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_image_size_control() {
		$this->skin->add_group_control(
			'image-size',
			[
				'name' => 'post_image_size',
				'default' => 'large',
			]
		);
	}

	/**
	 * Register settings controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_settings_controls() {
		$this->skin->add_control(
			'post_title_tag',
			[
				'label' => __( 'Title HTML Tag', 'raven' ),
				'type' => 'select',
				'default' => 'h3',
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
			]
		);

		$this->skin->add_control(
			'post_hover_effect',
			[
				'label' => __( 'Hover Effect', 'raven' ),
				'type' => 'raven_hover_effect',
			]
		);

		$this->skin->add_control(
			'show_image',
			[
				'label' => __( 'Featured Image', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_title',
			[
				'label' => __( 'Post Title', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_date',
			[
				'label' => __( 'Date', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_author',
			[
				'label' => __( 'Author', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_categories',
			[
				'label' => __( 'Categories', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_tags',
			[
				'label' => __( 'Tags', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_comments',
			[
				'label' => __( 'Comments', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_excerpt',
			[
				'label' => __( 'Excerpt', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);

		$this->skin->add_control(
			'show_button',
			[
				'label' => __( 'CTA Button', 'raven' ),
				'type' => 'switcher',
				'default' => 'yes',
				'label_on' => __( 'Show', 'raven' ),
				'label_off' => __( 'Hide', 'raven' ),
			]
		);
	}

	/**
	 * Register title controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
	 */
	protected function register_container_controls() {
		$this->skin->start_controls_section(
			'section_container',
			[
				'label' => __( 'Container', 'raven' ),
				'tab' => 'style',
			]
		);

		$this->skin->add_responsive_control(
			'post_padding',
			[
				'label' => __( 'Padding', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-post-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_align',
			[
				'label' => __( 'Alignment', 'raven' ),
				'type' => 'choose',
				'default' => '',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'raven' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'raven' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'raven' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->skin->start_controls_tabs( 'tabs_post' );

		$this->skin->start_controls_tab(
			'tab_post_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->skin->add_group_control(
			'raven-background',
			[
				'name' => 'post_background',
				'exclude' => [ 'image' ],
				'fields_options' => [
					'background' => [
						'label' => __( 'Background Color Type', 'raven' ),
					],
					'color' => [
						'label' => __( 'Background Color', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post',
			]
		);

		$this->skin->add_control(
			'post_border_heading',
			[
				'label' => __( 'Border', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->skin->add_control(
			'post_border_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'condition' => [
					$this->skin->get_control_id( 'post_border_border!' ) => '',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'border',
			[
				'name' => 'post_border',
				'placeholder' => '1px',
				'exclude' => [ 'color' ],
				'fields_options' => [
					'width' => [
						'label' => __( 'Border Width', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post',
			]
		);

		$this->skin->add_control(
			'post_border_radius',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_group_control(
			'box-shadow',
			[
				'name' => 'post_box_shadow',
				'selector' => '{{WRAPPER}} .raven-post',
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->start_controls_tab(
			'tab_post_hover',
			[
				'label' => __( 'Hover', 'raven' ),
			]
		);

		$this->skin->add_group_control(
			'raven-background',
			[
				'name' => 'post_background_hover',
				'exclude' => [ 'image' ],
				'fields_options' => [
					'background' => [
						'label' => __( 'Background Color Type', 'raven' ),
					],
					'color' => [
						'label' => __( 'Background Color', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post:hover',
			]
		);

		$this->skin->add_control(
			'post_border_heading_hover',
			[
				'label' => __( 'Border', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->skin->add_control(
			'post_border_color_hover',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'condition' => [
					$this->skin->get_control_id( 'post_border_hover_border!' ) => '',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'border',
			[
				'name' => 'post_border_hover',
				'placeholder' => '1px',
				'exclude' => [ 'color' ],
				'fields_options' => [
					'width' => [
						'label' => __( 'Border Width', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post:hover',
			]
		);

		$this->skin->add_control(
			'post_border_radius_hover',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_group_control(
			'box-shadow',
			[
				'name' => 'post_box_shadow_hover',
				'selector' => '{{WRAPPER}} .raven-post:hover',
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->end_controls_tabs();

		$this->skin->end_controls_section();
	}

	/**
	 * Register image controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
	 */
	protected function register_image_controls() {
		$this->skin->start_controls_section(
			'section_image',
			[
				'label' => __( 'Featured Image', 'raven' ),
				'tab' => 'style',
				'condition' => [
					$this->skin->get_control_id( 'show_image' ) => 'yes',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_image_height',
			[
				'label' => __( 'Image Ratio', 'raven' ),
				'type' => 'slider',
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0.1,
						'max' => 3,
						'step' => 0.1,
					],
				],
				'default' => [
					'size' => '0.63',
				],
				'tablet_default' => [
					'size' => '0.63',
				],
				'mobile_default' => [
					'size' => '0.63',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-image' => 'padding-bottom: calc( {{SIZE}} * 100% );',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_image_width',
			[
				'label' => __( 'Image Width', 'raven' ),
				'type' => 'slider',
				'size_units' => [ '%', 'px' ],
				'range' => [
					'%' => [
						'min' => 1,
						'max' => 100,
					],
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
				],
				'default' => [
					'size' => '100',
					'unit' => '%',
				],
				'tablet_default' => [
					'size' => '100',
					'unit' => '%',
				],
				'mobile_default' => [
					'size' => '100',
					'unit' => '%',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post:not(.raven-post-inline) .raven-post-image, {{WRAPPER}} .raven-post-inline .raven-post-image-wrap' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_control(
			'post_image_position',
			[
				'label' => __( 'Image Position', 'raven' ),
				'type' => 'select',
				'default' => 'top',
				'options' => [
					'left' => __( 'Left', 'raven' ),
					'right' => __( 'Right', 'raven' ),
					'top' => __( 'Top', 'raven' ),
				],
				'condition' => [
					$this->skin->get_control_id( 'layout' ) => 'grid',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_image_spacing',
			[
				'label' => __( 'Spacing', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-post-image-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} [data-mirrored] .raven-post-inline-left .raven-post-image-wrap' => 'padding: {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
					'{{WRAPPER}} [data-mirrored] .raven-post-inline-right .raven-post-image-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_control(
			'post_image_align',
			[
				'label' => __( 'Alignment', 'raven' ),
				'type' => 'choose',
				'label_block' => false,
				'default' => '',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'raven' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'raven' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'raven' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-image-wrap' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					$this->skin->get_control_id( 'post_image_position' ) => 'top',
				],
			]
		);

		$this->skin->start_controls_tabs( 'tabs_post_image' );

		$this->skin->start_controls_tab(
			'tab_post_image_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->skin->add_responsive_control(
			'post_image_opacity',
			[
				'label' => __( 'Opacity', 'raven' ),
				'type' => 'slider',
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1,
						'step' => 0.01,
					],
				],
				'default' => [
					'size' => '1',
				],
				'tablet_default' => [
					'size' => '1',
				],
				'mobile_default' => [
					'size' => '1',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-image img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'raven-background',
			[
				'name' => 'post_image_overlay',
				'exclude' => [ 'image' ],
				'fields_options' => [
					'background' => [
						'label' => __( 'Overlay Color Type', 'raven' ),
					],
					'color' => [
						'label' => __( 'Overlay Color', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-image-overlay',
			]
		);

		$this->skin->add_control(
			'post_image_border_heading',
			[
				'label' => __( 'Border', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->skin->add_control(
			'post_image_border_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-image' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'border',
			[
				'name' => 'post_image_border',
				'placeholder' => '1px',
				'exclude' => [ 'color' ],
				'fields_options' => [
					'width' => [
						'label' => __( 'Border Width', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-image',
			]
		);

		$this->skin->add_control(
			'post_image_border_radius',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->start_controls_tab(
			'tab_post_image_hover',
			[
				'label' => __( 'Hover', 'raven' ),
			]
		);

		$this->skin->add_responsive_control(
			'hover_post_image_opacity',
			[
				'label' => __( 'Opacity', 'raven' ),
				'type' => 'slider',
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1,
						'step' => 0.01,
					],
				],
				'default' => [
					'size' => '1',
				],
				'tablet_default' => [
					'size' => '1',
				],
				'mobile_default' => [
					'size' => '1',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-image:hover img' => 'opacity: {{SIZE}};',
					'{{WRAPPER}} .raven-post-inside:hover .raven-post-image img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'raven-background',
			[
				'name' => 'hover_post_image_overlay',
				'exclude' => [ 'image' ],
				'fields_options' => [
					'background' => [
						'label' => __( 'Overlay Color Type', 'raven' ),
					],
					'color' => [
						'label' => __( 'Overlay Color', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-image:hover .raven-post-image-overlay',
			]
		);

		$this->skin->add_control(
			'post_image_border_heading_hover',
			[
				'label' => __( 'Border', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->skin->add_control(
			'post_image_border_color_hover',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-image:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'border',
			[
				'name' => 'post_image_border_hover',
				'placeholder' => '1px',
				'exclude' => [ 'color' ],
				'fields_options' => [
					'width' => [
						'label' => __( 'Border Width', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-image:hover',
			]
		);

		$this->skin->add_control(
			'post_image_border_radius_hover',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-image:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->end_controls_tabs();

		$this->skin->end_controls_section();
	}

	/**
	 * Register title controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_title_controls() {
		$this->skin->start_controls_section(
			'section_title',
			[
				'label' => __( 'Post Title', 'raven' ),
				'tab' => 'style',
				'condition' => [
					$this->skin->get_control_id( 'show_title' ) => 'yes',
				],
			]
		);

		$this->skin->add_group_control(
			'typography',
			[
				'name' => 'post_title_typography',
				'scheme' => '1',
				'selector' => '{{WRAPPER}} .raven-post-title',
			]
		);

		$this->skin->add_responsive_control(
			'post_title_spacing',
			[
				'label' => __( 'Spacing', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-post-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_title_align',
			[
				'label' => __( 'Alignment', 'raven' ),
				'type' => 'choose',
				'default' => '',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'raven' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'raven' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'raven' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-title' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->skin->start_controls_tabs( 'tabs_post_title' );

		$this->skin->start_controls_tab(
			'tab_title_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->skin->add_control(
			'post_title_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} .raven-post-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->start_controls_tab(
			'tab_title_hover',
			[
				'label' => __( 'Hover', 'raven' ),
			]
		);

		$this->skin->add_control(
			'post_title_color_hover',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-title:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .raven-post-title:hover a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->end_controls_tabs();

		$this->skin->end_controls_section();
	}

	/**
	 * Register meta controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_meta_controls() {
		// Manually add `_skin` test condition.
		$conditions = [
			'relation' => 'and',
			'terms' => [
				[
					'relation' => 'or',
					'terms' => [
						[
							'name' => $this->skin->get_control_id( 'show_categories' ),
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => $this->skin->get_control_id( 'show_tags' ),
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => $this->skin->get_control_id( 'show_author' ),
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => $this->skin->get_control_id( 'show_date' ),
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => $this->skin->get_control_id( 'show_comments' ),
							'operator' => '===',
							'value' => 'yes',
						],
					],
				],
				[
					'name' => '_skin',
					'operator' => '===',
					'value' => $this->skin->get_id(),
				],
			],
		];

		$this->skin->start_controls_section(
			'section_meta',
			[
				'label' => __( 'Meta', 'raven' ),
				'tab' => 'style',
				'conditions' => $conditions,
			]
		);

		$this->skin->add_group_control(
			'typography',
			[
				'name' => 'post_meta_typography',
				'scheme' => '3',
				'conditions' => $conditions,
				'selector' => '{{WRAPPER}} .raven-post-meta',
			]
		);

		$this->skin->add_control(
			'post_meta_divider',
			[
				'label' => __( 'Meta Divider', 'raven' ),
				'type' => 'text',
				'default' => '/',
				'conditions' => $conditions,
			]
		);

		$this->skin->add_responsive_control(
			'post_meta_divider_spacing',
			[
				'label' => __( 'Divider Spacing', 'raven' ),
				'type' => 'slider',
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'conditions' => $conditions,
				'selectors' => [
					'{{WRAPPER}} .raven-post-meta-divider' => 'margin-right: {{SIZE}}{{UNIT}}; margin-left: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_meta_spacing',
			[
				'label' => __( 'Spacing', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'conditions' => $conditions,
				'selectors' => [
					'{{WRAPPER}} .raven-post-meta' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_meta_align',
			[
				'label' => __( 'Alignment', 'raven' ),
				'type' => 'choose',
				'default' => '',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'raven' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'raven' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'raven' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'conditions' => $conditions,
				'selectors' => [
					'{{WRAPPER}} .raven-post-meta' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->skin->start_controls_tabs( 'tabs_post_meta' );
		$this->skin->start_controls_tab(
			'tab_post_meta_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->skin->add_control(
			'post_meta_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'conditions' => $conditions,
				'selectors' => [
					'{{WRAPPER}} .raven-post-meta' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_control(
			'post_meta_links_color',
			[
				'label' => __( 'Links Color', 'raven' ),
				'type' => 'color',
				'conditions' => $conditions,
				'selectors' => [
					'{{WRAPPER}} .raven-post-meta a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->start_controls_tab(
			'tab_post_meta_hover',
			[
				'label' => __( 'Hover', 'raven' ),
			]
		);

		$this->skin->add_control(
			'post_meta_color_hover',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'conditions' => $conditions,
				'selectors' => [
					'{{WRAPPER}} .raven-post-meta:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_control(
			'post_meta_links_color_hover',
			[
				'label' => __( 'Links Color', 'raven' ),
				'type' => 'color',
				'conditions' => $conditions,
				'selectors' => [
					'{{WRAPPER}} .raven-post-meta:hover a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->end_controls_tabs();

		$this->skin->end_controls_section();
	}

	/**
	 * Register excerpt controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_excerpt_controls() {
		$this->skin->start_controls_section(
			'section_excerpt',
			[
				'label' => __( 'Excerpt', 'raven' ),
				'tab' => 'style',
				'condition' => [
					$this->skin->get_control_id( 'show_excerpt' ) => 'yes',
				],
			]
		);

		$this->skin->add_control(
			'excerpt_length',
			[
				'label' => __( 'Excerpt Length', 'raven' ),
				'type' => 'slider',
				'size_units' => [ 'px' ],
				'default' => [
					'size' => 30,
				],
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 150,
						'step' => 1,
					],
				],
			]
		);

		$this->skin->add_group_control(
			'typography',
			[
				'name' => 'post_excerpt_typography',
				'scheme' => '3',
				'selector' => '{{WRAPPER}} .raven-post-excerpt',
			]
		);

		$this->skin->add_responsive_control(
			'post_excerpt_spacing',
			[
				'label' => __( 'Spacing', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-post-excerpt' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_excerpt_align',
			[
				'label' => __( 'Alignment', 'raven' ),
				'type' => 'choose',
				'default' => '',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'raven' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'raven' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'raven' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-excerpt' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->skin->start_controls_tabs( 'tabs_post_excerpt' );

		$this->skin->start_controls_tab(
			'tab_post_excerpt_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->skin->add_control(
			'post_excerpt_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-excerpt' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->start_controls_tab(
			'tab_post_excerpt_hover',
			[
				'label' => __( 'Hover', 'raven' ),
			]
		);

		$this->skin->add_control(
			'post_excerpt_color_hover',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-excerpt:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->end_controls_tabs();

		$this->skin->end_controls_section();
	}

	/**
	 * Register button controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
	 */
	protected function register_button_controls() {
		$this->skin->start_controls_section(
			'section_button',
			[
				'label' => __( 'CTA Button', 'raven' ),
				'tab' => 'style',
				'condition' => [
					$this->skin->get_control_id( 'show_button' ) => 'yes',
				],
			]
		);

		$this->skin->add_control(
			'post_button_text',
			[
				'label' => __( 'Text', 'raven' ),
				'type' => 'text',
				'default' => __( 'Read More', 'raven' ),
			]
		);

		$this->skin->add_responsive_control(
			'post_button_width',
			[
				'label' => __( 'Width', 'raven' ),
				'type' => 'slider',
				'size_units' => [ 'px', 'em', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'tablet_default' => [
					'unit' => 'px',
				],
				'mobile_default' => [
					'unit' => 'px',
				],
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-button' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_button_height',
			[
				'label' => __( 'Height', 'raven' ),
				'type' => 'slider',
				'size_units' => [ 'px', 'em' ],
				'default' => [
					'unit' => 'px',
				],
				'tablet_default' => [
					'unit' => 'px',
				],
				'mobile_default' => [
					'unit' => 'px',
				],
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-button' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_button_spacing',
			[
				'label' => __( 'Spacing', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .raven-post-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_responsive_control(
			'post_button_align',
			[
				'label' => __( 'Alignment', 'raven' ),
				'type' => 'choose',
				'prefix_class' => 'raven%s-button-align-',
				'default' => '',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'raven' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'raven' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'raven' ),
						'icon' => 'fa fa-align-right',
					],
					'justify' => [
						'title' => __( 'Justified', 'raven' ),
						'icon' => 'fa fa-align-justify',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-read-more' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->skin->start_controls_tabs( 'tabs_button' );

		$this->skin->start_controls_tab(
			'tabs_button_normal',
			[
				'label' => __( 'Normal', 'raven' ),
			]
		);

		$this->skin->add_control(
			'post_button_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'typography',
			[
				'name' => 'post_button_typography',
				'scheme' => '3',
				'selector' => '{{WRAPPER}} .raven-post-button',
			]
		);

		$this->skin->add_group_control(
			'raven-background',
			[
				'name' => 'post_button_background',
				'exclude' => [ 'image' ],
				'fields_options' => [
					'background' => [
						'label' => __( 'Background Color Type', 'raven' ),
					],
					'color' => [
						'label' => __( 'Background Color', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-button',
			]
		);

		$this->skin->add_control(
			'post_button_border_heading',
			[
				'label' => __( 'Border', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->skin->add_control(
			'post_button_border_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'condition' => [
					$this->skin->get_control_id( 'post_button_border_border!' ) => '',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-button' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'border',
			[
				'name' => 'post_button_border',
				'placeholder' => '1px',
				'exclude' => [ 'color' ],
				'fields_options' => [
					'width' => [
						'label' => __( 'Border Width', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-button',
			]
		);

		$this->skin->add_control(
			'post_button_border_radius',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_group_control(
			'box-shadow',
			[
				'name' => 'post_button_box_shadow',
				'selector' => '{{WRAPPER}} .raven-post-button',
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->start_controls_tab(
			'tabs_button_hover',
			[
				'label' => __( 'Hover', 'raven' ),
			]
		);

		$this->skin->add_control(
			'hover_post_button_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'selectors' => [
					'{{WRAPPER}} .raven-post-button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'typography',
			[
				'name' => 'hover_post_button_typography',
				'scheme' => '3',
				'selector' => '{{WRAPPER}} .raven-post-button:hover',
			]
		);

		$this->skin->add_group_control(
			'raven-background',
			[
				'name' => 'hover_post_button_background',
				'exclude' => [ 'image' ],
				'fields_options' => [
					'background' => [
						'label' => __( 'Background Color Type', 'raven' ),
					],
					'color' => [
						'label' => __( 'Background Color', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-button:hover',
			]
		);

		$this->skin->add_control(
			'hover_post_button_border_heading',
			[
				'label' => __( 'Border', 'raven' ),
				'type' => 'heading',
				'separator' => 'before',
			]
		);

		$this->skin->add_control(
			'hover_post_button_border_color',
			[
				'label' => __( 'Color', 'raven' ),
				'type' => 'color',
				'condition' => [
					$this->skin->get_control_id( 'hover_post_button_border_border!' ) => '',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->skin->add_group_control(
			'border',
			[
				'name' => 'hover_post_button_border',
				'placeholder' => '1px',
				'exclude' => [ 'color' ],
				'fields_options' => [
					'width' => [
						'label' => __( 'Border Width', 'raven' ),
					],
				],
				'selector' => '{{WRAPPER}} .raven-post-button:hover',
			]
		);

		$this->skin->add_control(
			'hover_post_button_border_radius',
			[
				'label' => __( 'Border Radius', 'raven' ),
				'type' => 'dimensions',
				'size_units' => [ 'px', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .raven-post-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->skin->add_group_control(
			'box-shadow',
			[
				'name' => 'hover_post_button_box_shadow',
				'selector' => '{{WRAPPER}} .raven-post-button:hover',
			]
		);

		$this->skin->end_controls_tab();

		$this->skin->end_controls_tabs();

		$this->skin->end_controls_section();
	}

	/**
	 * Get render by stack.
	 *
	 * Use to get render in a stack list format.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @param array $stack_list List of function names.
	 */
	protected function get_render_stack( $stack_list = [] ) {
		$stack_render = [];

		foreach ( $stack_list as $stack_item ) {
			$func_name = 'get_render_' . $stack_item;

			$to_render = $this->$func_name();

			if ( ! empty( $to_render ) ) {
				$stack_render[] = $to_render;
			}
		}

		return $stack_render;
	}

	/**
	 * Render the post.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param object $instance Instance of either Widget or Skin.
	 */
	public function render_post( $instance ) {
		$this->skin = $instance;

		$layout = $this->skin->get_instance_value( 'layout' );

		$show_image = $this->skin->get_instance_value( 'show_image' );

		$image_position = $this->skin->get_instance_value( 'post_image_position' );

		$hover_effect = $this->skin->get_instance_value( 'post_hover_effect' );

		$item_classes = [
			'masonry' === $layout ? 'raven-masonry-item' : 'raven-grid-item',
			'raven-post-item',
		];

		$item_classes = array_merge( $item_classes, get_post_class() );

		$post_classes = [ 'raven-post' ];

		if ( 'grid' === $layout && 'yes' === $show_image && 'top' !== $image_position ) {
			$post_classes[] = 'raven-post-inline raven-post-inline-' . $image_position;
		}

		if ( ! empty( $hover_effect ) ) {
			$post_classes[] = 'elementor-animation-' . $hover_effect;
		}

		?>
		<div class="<?php echo esc_attr( implode( ' ', $item_classes ) ); ?>">
			<div class="<?php echo esc_attr( implode( ' ', $post_classes ) ); ?>">
				<?php $this->render_image(); ?>

				<div class="raven-post-content">
					<?php
					$this->render_title();
					$this->render_meta();
					$this->render_excerpt();
					$this->render_button();
					?>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render the post image.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render_image() {
		if ( ! $this->skin->get_instance_value( 'show_image' ) ) {
			return;
		}

		$settings = [
			'image_size' => $this->skin->get_instance_value( 'post_image_size_size' ),
			'image' => [
				'id' => get_post_thumbnail_id(),
			],
			'image_custom_dimension' => $this->skin->get_instance_value( 'post_image_size_custom_dimension' ),
		];

		$image_html = \Elementor\Group_Control_Image_Size::get_attachment_image_html( $settings );

		if ( empty( $image_html ) ) {
			return;
		}

		$classes = [
			'raven-post-image',
		];

		$html_tag = 'a';

		if ( 'grid' === $this->skin->get_instance_value( 'layout' ) ) {
			$classes[] = 'raven-image-fit';
		}

		?>
		<div class="raven-post-image-wrap">
			<?php
			printf(
				'<%1$s class="%2$s" %3$s>%4$s <span class="raven-post-image-overlay"></span></%1$s>',
				$html_tag,
				implode( ' ', $classes ),
				( 'a' === $html_tag ) ? 'href="' . get_permalink() . '"' : '',
				$image_html
			);
			?>
		</div>
		<?php
	}

	/**
	 * Render the post title.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render_title() {
		if ( ! $this->skin->get_instance_value( 'show_title' ) ) {
			return;
		}

		printf(
			'<%1$s class="raven-post-title"><a class="raven-post-title-link" href="%2$s">%3$s</a></%1$s>',
			$this->skin->get_instance_value( 'post_title_tag' ),
			get_permalink(),
			get_the_title()
		);
	}

	/**
	 * Render the post meta.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render_meta() {
		// We can modify this to re-order the meta stack.
		$meta_list = [
			'date',
			'author',
			'categories',
			'tags',
			'comments',
		];

		$meta_stack = $this->get_render_stack( $meta_list );

		if ( empty( $meta_stack ) ) {
			return;
		}

		$meta_html = implode( $this->get_render_divider(), $meta_stack );

		?>
		<div class="raven-post-meta"><?php echo $meta_html; ?></div>
		<?php
	}

	/**
	 * Render the post meta divider.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_render_divider() {
		if ( ! $this->skin->get_instance_value( 'post_meta_divider' ) ) {
			return PHP_EOL;
		}

		return PHP_EOL . '<span class="raven-post-meta-divider">' . $this->skin->get_instance_value( 'post_meta_divider' ) . '</span>' . PHP_EOL;
	}

	/**
	 * Render the post meta date.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_render_date() {
		if ( ! $this->skin->get_instance_value( 'show_date' ) ) {
			return;
		}

		return '<a class="raven-post-meta-item raven-post-date" href="' . get_permalink() . '" rel="bookmark">' . get_the_date() . '</a>';
	}

	/**
	 * Render the post meta author.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_render_author() {
		if ( ! $this->skin->get_instance_value( 'show_author' ) ) {
			return;
		}

		return '<a class="raven-post-meta-item raven-post-author" href="' . get_author_posts_url( get_the_author_meta( 'ID' ) ) . '">' . get_the_author() . '</a>';
	}

	/**
	 * Render the post meta categories.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_render_categories() {
		$post_type = get_post_type();

		if ( ! in_array( $post_type, [ 'post', 'portfolio' ], true ) || ! $this->skin->get_instance_value( 'show_categories' ) ) {
			return;
		}

		$taxonomy = 'category';

		if ( 'portfolio' === $post_type ) {
			$taxonomy = 'portfolio_category';
		}

		$categories_list = get_the_term_list( get_the_ID(), $taxonomy, '', ', ', '' );

		if ( empty( $categories_list ) ) {
			return;
		}

		return sprintf( '<span class="raven-post-meta-item raven-post-categories">%1$s</span>', $categories_list );
	}

	/**
	 * Render the post meta tags.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_render_tags() {
		$post_type = get_post_type();

		if ( ! in_array( $post_type, [ 'post', 'portfolio' ], true ) || ! $this->skin->get_instance_value( 'show_tags' ) ) {
			return;
		}

		$taxonomy = 'post_tag';

		if ( 'portfolio' === $post_type ) {
			$taxonomy = 'portfolio_tag';
		}

		$tags_list = get_the_term_list( get_the_ID(), $taxonomy, '', ', ', '' );

		if ( empty( $tags_list ) ) {
			return;
		}

		return sprintf( '<span class="raven-post-meta-item raven-post-tags">%1$s</span>', $tags_list );
	}

	/**
	 * Render the post comments.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_render_comments() {
		if ( ! $this->skin->get_instance_value( 'show_comments' ) ) {
			return;
		}

		return '<a class="raven-post-meta-item raven-post-comments" href="' . get_permalink() . '#comments" rel="bookmark">' . get_comments_number_text() . '</a>';
	}

	/**
	 * Render the post excerpt.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render_excerpt() {
		if ( ! $this->skin->get_instance_value( 'show_excerpt' ) ) {
			return;
		}

		?>
		<div class="raven-post-excerpt"><?php echo get_the_excerpt(); ?></div>
		<?php
	}

	/**
	 * Render the post button.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render_button() {
		if ( ! $this->skin->get_instance_value( 'show_button' ) ) {
			return;
		}

		?>
		<div class="raven-post-read-more">
			<a class="raven-post-button" href="<?php the_permalink(); ?>"><span class="raven-post-button-text"><?php echo $this->skin->get_instance_value( 'post_button_text' ); ?></span></a>
		</div>
		<?php
	}

	/**
	 * Render the post render icon.
	 *
	 * Currently supports Dashicons from WordPress.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_render_icon() {
		if ( ! $this->skin->get_instance_value( 'show_image' ) ) {
			return;
		}

		$post_type = get_post_type_object( get_post_type() );

		if ( empty( $post_type ) ) {
			return;
		}

		$icon_class = [
			'raven-post-icon',
			'dashicons-before',
		];

		$icon_class[] = false !== strpos( $post_type->menu_icon, 'dashicon' ) ? $post_type->menu_icon : 'dashicons-admin-post';

		$icon_html = sprintf( '<span class="%1$s"></span>', implode( ' ', $icon_class ) );

		return $icon_html;
	}
}
